#include "ald_plugin.h"

#include <gio/gio.h>
#include <glib-unix.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/eventfd.h>
#include <pthread.h>

#include "app_iface/generated/ald-dbus-interface.h"


#define printf (void)

#define ALD_BUS_NAME              	"com.adit.de.ALD"
#define ALD_LEVEL_STATUS_OBJECT      	"/com/adit/de/ALD/level_status"

//------------------------------------ private structs ---------------------------------------------------------------
typedef enum ald_plugin_state_t
{
	UNINITIALIZED,
	INITIALIZED_DISCONNECTED,
	INITIALIZED_WAITING_FOR_CONNECTION,
	INITIALIZED_CONNECTED
} ald_plugin_state_t;

typedef enum glib_dbus_events_t
{
	_NOEVENTS=0,
	ALD_APPEARED_EVENT=0x01,
	ALD_VANISHED_EVENT=0x02,
	ALD_LEVEL_CHANGED_EVENT=0x04
} glib_dbus_events_t;
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------ private attributes ------------------------------------------------------------
static int event_fd=-1;

static unsigned int ald_default_level=0;
static unsigned int changed_level=0;

static ald_plugin_state_t state=UNINITIALIZED;

static const ald_plugin_callbacks_t *registered_callbacks=NULL;

//pending_glib_dbus_events is an event mask that is used to channel events received by the g_mainloop to the daemon.
//The g_mainloop fills up an event mask (mutexed) and signals the daemon side via eventfd that new events are available.
//The daemon side reads out the event mask and reacts with its own thread repsectively.
static unsigned char pending_glib_dbus_events=_NOEVENTS;
static pthread_mutex_t pending_events_mutex = PTHREAD_MUTEX_INITIALIZER;

static GMainLoop *g_main_loop=NULL;

static Level_status *ald_level_interface=NULL;
static guint busname_watcher_id;
static pthread_t ald_plugin_thread;
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------ private functions -------------------------------------------------------------
static void ald_plugin_do_connect(void* dataPtr, void* dataPtr2);
static void ald_plugin_do_disconnect(bool auto_reconnect,void * data_ptr, void * data_ptr2);

//------------------------------------ connect / disconnect ----------------------------------------------------------
static void ald_plugin_deferred_connect();
static void ald_plugin_disconnect(void * data_ptr, void * data_ptr2);

static void ald_plugin_ald_appeared(GDBusConnection *connection, const gchar *name,
		const gchar *name_owner, gpointer user_data);
static void ald_plugin_ald_vanished(GDBusConnection *connection, const gchar *name, gpointer user_data);
static void ald_plugin_on_level_changed(Level_status *iface, guint16 new_level);
static void ald_plugin_emit_signal(glib_dbus_events_t event);

static int ald_plugin_start_mainloop();
static void *ald_plugin_thread_func(void *params);
static gboolean ald_plugin_queue_mainloop_quit (gpointer user_data);
static int ald_plugin_quit_mainloop(void);


//------------------------------------ init / deinit -----------------------------------------------------------------
int ald_plugin_init(const ald_plugin_callbacks_t *callbacks, unsigned int default_level)
{
    fprintf(stdout,"ALD Plugin  - Initializing Plugin\n");
    int retVal = 0;

    //ignore if called twice
    if (state != UNINITIALIZED)
	return -1;
    
    if(callbacks != NULL)
    {
        registered_callbacks=callbacks;
    }
    else
    {
        fprintf(stdout,"ALD_PLUGIN - Init exited, callback structure is NULL\n");
        return -1;
    }
    
    //Initailise an eventfd for communication between GMaimLoop Thread and external Mainloop e.g DLT Daemon's main loop
    event_fd = eventfd(0,EFD_NONBLOCK);
	
    if(event_fd == -1)
    {   
        printf("ALD_PLUGIN - Init Failed to create eventfd:\n"/*,strerror(errno)*/);
        return -1;
    }

    ald_default_level=default_level;

#if !GLIB_CHECK_VERSION (2, 35, 0)
    g_type_init();
#endif
    //start the gMainLoop Thread.
    retVal = ald_plugin_start_mainloop();
    if(retVal != 0)
    {
        close(event_fd);
        fprintf(stdout,"ALD_PLUGIN - Init Failed to start_mainloop\n");
        return -1;
    }
    state=INITIALIZED_DISCONNECTED;
    //printf("ALD_PLUGIN - Initialized with default level %d \n",default_level);

    /* Here plugin does not connect directly, rather watches the bus name of the ALD and wait for it to appear.
     Once ALD is available, plugin shall connect */
    ald_plugin_deferred_connect();
    fprintf(stdout,"ALD Plugin  - FINISHED Initializing Plugin\n");
    return 0;
}


int ald_plugin_deinit(void *ptr, void * ptr2)
{
    int retVal = 0;

    if((ptr == NULL) || (ptr2 == NULL))
    {
        fprintf(stdout,"ALD_PLUGIN - Inavlid input parameters (NULL) in ald_plugin_deinit.\n");
        return -1;
    }
	
    //ignore if called twice
    if (state == UNINITIALIZED)
	return -1;

    if ((state == INITIALIZED_CONNECTED) || (state == INITIALIZED_WAITING_FOR_CONNECTION))
        ald_plugin_disconnect(ptr,ptr2);

    retVal = close(event_fd);
    event_fd=0;

    ald_default_level=0;

    registered_callbacks=NULL;

    retVal = ald_plugin_quit_mainloop();

    g_main_loop_unref(g_main_loop);
    g_main_loop=NULL;

    state=UNINITIALIZED;
    fprintf(stdout,"ALD_PLUGIN - DeInitialized plugin\n");

    return retVal;
}
//--------------------------------------------------------------------------------------------------------------------


//------------------------------------ connect / disconnect ----------------------------------------------------------

void ald_plugin_deferred_connect()
{
    //we only accept this call when we have been initialized but are not connected or still waiting for a connection
    if (state != INITIALIZED_DISCONNECTED)
        return;
    
    state=INITIALIZED_WAITING_FOR_CONNECTION;
    /*Here plugin does not connect directly, rather watches the bus name of the ALD and wait for it to appear. The
    g_bus_watch_name will send an initial event informing about the current state. */
    fprintf(stdout,"ALD_PLUGIN - Trying to Connect by watching for ALD bus name. Then later connect\n");
    busname_watcher_id=g_bus_watch_name (G_BUS_TYPE_SYSTEM,ALD_BUS_NAME,G_BUS_NAME_WATCHER_FLAGS_NONE,
			ald_plugin_ald_appeared,ald_plugin_ald_vanished,NULL,NULL);
}

void ald_plugin_disconnect(void *ptr, void * ptr2)
{
    if ((state != INITIALIZED_CONNECTED) && (state != INITIALIZED_WAITING_FOR_CONNECTION))
	return;

    g_bus_unwatch_name(busname_watcher_id);
    
    if((ptr != NULL) && (ptr2 != NULL))
    {
       ald_plugin_do_disconnect(false,ptr,ptr2);
    }
}

static void ald_plugin_do_connect(void *ptr, void * ptr2)
{
    GError *err = NULL;

    if (state!=INITIALIZED_WAITING_FOR_CONNECTION)
		return;

    if((ptr == NULL) || (ptr2 == NULL))
    {
        fprintf(stdout,"ALD_PLUGIN - Inavlid input parameters (NULL) in ald_plugin_do_connect.\n");
        
        fprintf(stdout,"ALD_PLUGIN - Connect to ALD exited!!\n");
        return;
    }

    fprintf(stdout,"ALD_PLUGIN - Connecting with ald ...\n");
    ald_level_interface= level_status_proxy_new_for_bus_sync(G_BUS_TYPE_SYSTEM,G_DBUS_PROXY_FLAGS_NONE, ALD_BUS_NAME, ALD_LEVEL_STATUS_OBJECT,
			NULL,&err);
    if (err != NULL)
    {
        //printf("Unable to connect the ALD: %s\n",  err->message);
        g_error_free(err);

        //we don't change the state here, we try it again when the daemon appears again
        return;
    }

    g_signal_connect(ald_level_interface, "level-change-done", G_CALLBACK(ald_plugin_on_level_changed), NULL);
    state=INITIALIZED_CONNECTED;
    fprintf(stdout,"ALD_PLUGIN - Connection established.\n");

    //calling callback of the daemon
    if ((registered_callbacks!=NULL) && (registered_callbacks->connection_clbk!=NULL))
    {
	registered_callbacks->connection_clbk(ptr,ptr2);
    }
}

static void ald_plugin_do_disconnect(bool auto_reconnect,void *ptr, void * ptr2)
{
    if((ptr == NULL) || (ptr2 == NULL))
    {
        fprintf(stdout,"ALD_PLUGIN - Inavlid input parameters (NULL) in ald_plugin_do_disconnect.\n");
        return;
    }

    if (state!=INITIALIZED_CONNECTED && state!=INITIALIZED_WAITING_FOR_CONNECTION)
    {
        if ((registered_callbacks!=NULL) && (registered_callbacks->disConnection_clbk!=NULL))
        {
            registered_callbacks->disConnection_clbk(ptr,ptr2);
        }  
	return;
    }

    if (state==INITIALIZED_CONNECTED && ald_level_interface!=NULL)
    {
        g_object_unref(ald_level_interface);
        ald_level_interface=NULL;

        //calling callback of the daemon
        if ((registered_callbacks!=NULL) && (registered_callbacks->disConnection_clbk!=NULL))
        {
            registered_callbacks->disConnection_clbk(ptr,ptr2);
        }  
    }

    if (auto_reconnect)
    {
        state=INITIALIZED_WAITING_FOR_CONNECTION;
    }
    else
    {
        state=INITIALIZED_DISCONNECTED;
    }
}
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------ event handling ----------------------------------------------------------------
int ald_plugin_get_pollfd(void)
{
    return event_fd;
}

int ald_plugin_dispatch(void *ptr, void *ptr2)
{
    uint64_t tmp;
    int retVal = 0;
   
    if((ptr == NULL) || (ptr2 == NULL))
    {
        fprintf(stdout,"ALD_PLUGIN - Inavlid input parameters (NULL) in ald_plugin_dispatch.\n");
        return -1;
    }
   
    //nothing to do if we are not initialized
    if (state==UNINITIALIZED)
        return -1;

    //reset the eventfd
    if (read(event_fd,&tmp,sizeof(uint64_t))==0)
    {
       // WHAT TO DO HERE
    }


    retVal = pthread_mutex_lock(&pending_events_mutex);
    //lets prevent that the plugin thread sets new events while we are dispatching the current ones
    if(retVal == 0)
    {
        if ((pending_glib_dbus_events & ALD_APPEARED_EVENT) != 0)
            //state checks are done in do_ald_plugin_connect()
	    ald_plugin_do_connect(ptr,ptr2);

        if ((pending_glib_dbus_events & ALD_LEVEL_CHANGED_EVENT) != 0 && state == INITIALIZED_CONNECTED)
        {
	    if ((registered_callbacks!=NULL) &&(registered_callbacks->calculate_log_level_clbk!=NULL))
	                      registered_callbacks->calculate_log_level_clbk(ald_plugin_get_security_level(0),ptr,ptr2);

        }

        if ((pending_glib_dbus_events & ALD_VANISHED_EVENT) != 0)
        //state checks are done in do_ald_plugin_disconnect()
        ald_plugin_do_disconnect(true,ptr,ptr2);

        //reset the event mask
        pending_glib_dbus_events=_NOEVENTS;

        retVal = pthread_mutex_unlock(&pending_events_mutex);  
    }
    /* PRQA: Lint Message 456: Mutex Lock and Unlock are handled correctly */ 

    /* PRQA: Lint Message 454: Mutex Lock and Unlock are handled correctly */ 
   
    return retVal; /*lint !e454 !e456 */
}

static int ald_plugin_start_mainloop()
{
    int retVal = 0;
    //NULL: mainloop with default glib main_context, FALSE: not started yet
    g_main_loop = g_main_loop_new(NULL, FALSE);

    if(g_main_loop != NULL)
    {
        retVal = pthread_create(&ald_plugin_thread,NULL,ald_plugin_thread_func,NULL);
        if(retVal != 0)
            printf("ALD_PLUGIN - Failed to create thread in start_mainloop\n");
    }

    return retVal;
}

static gboolean ald_plugin_queue_mainloop_quit (gpointer user_data)
{
	(void)user_data;

	g_main_loop_quit(g_main_loop);

	return G_SOURCE_REMOVE;
}

static int ald_plugin_quit_mainloop(void)
{
    int retVal = 0;
    
    //glib is thread safe so we don't need to take care with calling glib functions with another thread
    /* To add the mainloop quit as function to be called whenever there are no higher priority events pending
     * to the default main loop. This method is used to avoid calling quit before even mainloop is started.
     */
    g_idle_add(ald_plugin_queue_mainloop_quit,NULL);

    //wait for the thread to die
    retVal = pthread_join(ald_plugin_thread,NULL);
    if(retVal != 0)
        printf("ALD_PLUGIN - Failed to join thread in quit_mainloop\n");

    return retVal;
}
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------ properties --------------------------------------------------------------------
unsigned int ald_plugin_get_security_level(bool onConnection )
{
    if (ald_level_interface!= NULL)
    {
        if(onConnection == true)
           return level_status_get_level(ald_level_interface);
        else
	 return changed_level;
    }
    else
    {   
        printf("ALD_PLUGIN - default level\n"); 
        return ald_default_level;
    }
}

bool ald_plugin_is_connected()
{
    return (state==INITIALIZED_CONNECTED);
}
//--------------------------------------------------------------------------------------------------------------------




//--------------------------------------------------------------------------------------------------------------------
//                                        executed with glib thread
//--------------------------------------------------------------------------------------------------------------------
static void *ald_plugin_thread_func(void *params)
{
    (void)params;
    printf("ALD_PLUGIN - Starting g_main_loop.\n");

    if(g_main_loop != NULL)
        g_main_loop_run(g_main_loop);
    printf("ALD_PLUGIN - g_main_loop left.\n");
    return NULL;
}

static void ald_plugin_ald_appeared(GDBusConnection *connection, const gchar *name,
		const gchar *name_owner, gpointer user_data)
{
    printf("ALD_PLUGIN - Ald bus name available\n");
    connection = connection;
    name = name;
    user_data = user_data;
    name_owner = name_owner;
    ald_plugin_emit_signal(ALD_APPEARED_EVENT);
}

static void ald_plugin_ald_vanished(GDBusConnection *connection, const gchar *name, gpointer user_data)
{
    printf("ALD_PLUGIN - Ald disappeared.\n");

    connection = connection;
    name = name;
    user_data = user_data;
    ald_plugin_emit_signal(ALD_VANISHED_EVENT);
}

static void ald_plugin_on_level_changed(Level_status *iface, guint16 new_level)
{
   // unsigned int new_auth_level = 0;
    iface = iface;

    changed_level = new_level;
    fprintf(stdout,"ALD_PLUGIN - ald_plugin_on_level_changed %d\n",new_level);
 
    ald_plugin_emit_signal(ALD_LEVEL_CHANGED_EVENT);
}

static void ald_plugin_emit_signal(glib_dbus_events_t event)
{
    uint64_t tmp=1;
    
    fprintf(stdout,"ALD_PLUGIN - ald_plugin_emit_signal %d\n",event);
    //lets prevent that events are currently dispatched while we are writing in new ones
    pthread_mutex_lock(&pending_events_mutex);

    pending_glib_dbus_events |= (unsigned char)event;
    if (write(event_fd, &tmp, sizeof(uint64_t))==0)
    {
  
    }

    pthread_mutex_unlock(&pending_events_mutex);
}
//--------------------------------------------------------------------------------------------------------------------
